	TITLE	'PERSEPHONE MINI/MICROFLOPPY DISK CONTROLLER'
	SUBTTL	'DOCUMENTATION'

	LIST	-I,-M,-G,R

**	FUNCTION
*
*	This program is the controller for the Atari 1050 CR (cost
*	reduced) minifloppy disk drive and for a possible future
*	microfloppy disk drive.  It resides in the ROM of an
*	Intel 8050 microcontroller.  It's function is to accept
*	commands from the Atari serial bus and to execute them by
*	driving a Western Digital 1770 FDC chip.


**	MODS
*
*	Original Author Mike Barall 09/11/84


**	HARDWARE CONFIGURATION

*	The clock rate is 8 MHz, which gives a time of 1.875 us per
*	machine cycle.

*	P10 and P11 select a register in the 1770 chip:
*		P11 P10		Register
*		 0   0		Status or Command
*       	 0   1		Track
*		 1   0		Sector
*		 1   1		Data

*	P12 is connected to the INTRQ line of the 1770

*	P13 selects FM (1) or MFM (0)

*	P14 is connected to the 1770 Master Reset pin (0 = reset).
*	The controller holds P14 low for 500 us.

*	P15 is connected to the 1770 R/W pin (1 = read, 0 = write).

*	P16 is the side select output (0 = side 0, 1 = side 1).
*	The controller delays 500 us after changing sides.  P16 is
*	externally tied to ground if the drive has only one head.

*	P17 is connected to the serial bus data in line (drive to
*	computer).  This line is normally high.  Note: P17 is the
*	inverse of the SID line.

*	P20 through P24 have no external connections.  They are used by
*	the controller as 5 bits of RAM.

*	P25 has no external connection on a minifloppy and is externally
*	grounded on a microfloppy.  In the former case, P25 is used as
*	one bit of RAM.

*	P26 and P27 are connected to the device select switches.  They
*	select this drive's unit number, as follows:
*	P27 P26		Unit Number
*	 0   0	        1
*	 0   1		2
*	 1   0		3
*	 1   1		4

*	INT is connected to the serial bus command line.  This line is
*	normally high, and is low during the transmission of a
*	command frame.

*	T0 is connected to the serial bus data out line (computer to
*	drive).  Note: T0 is the inverse of the SOD line.

*	T1 is connected to the DRQ line of the 1770.  A high on this
*	line indicates that the 1770 data register is full (on a
*	read) or empty (on a write).

*	DB0-DB7 (the external data bus) are connected to the 1770
*	data bus.


**	MEMORY ALLOCATION

*	F0 = SIO fast mode enable/disable (1 = enabled).

*	F1 = Retry flag (1 = retry in progress).

*	P20, P21, P22 = Current activity:
*
*	P22 P21 P20
*	 0   0   0	Idle
*	 0   0   1	Get sector
*	 0   1   0	Put sector without verify
*	 0   1   1	Put sector with verify
*	 1   0   0	Write option table
*	 1   0   1	Miscellaneous disk operation
*	 1   1   0	Not used
*	 1   1   1	Other

*	P23 = Head position known flag (1 = position known).

*	P24, P25 = Current configuration:
*
*	P25 P24
*	 1   0		SS/SD (FM, 128 bytes per sector)
*	 1   1		SS/2D (MFM, 128 bytes per sector)
*	 0   0		SS/DD (MFM, 256 bytes per sector)
*	 0   1		DS/DD (MFM, 256 bytes per sector)
*
*	For microfloppies, only the SS/DD and DS/DD modes are possible,
*	and there are sixteen sectors per track.


**	COMMANDS

* 52	GET SECTOR
*
*	Read the specified logical sector.

* 50	PUT SECTOR WITHOUT VERIFY
*
*	Write the specified logical sector.

* 57	PUT SECTOR WITH VERIFY
*
*	Write the specified logical sector and then re-read it to
*	verify that it was written correctly.

* 53	STATUS
*
*	Return a 4-byte status block.

* 21	FORMAT
*
*	Format the diskette in the current configuration.

* 22	DUAL DENSITY FORMAT
*
*	Format the diskette in SS/2D mode.  Valid only when the drive
*	is configured to SS/SD or SS/2D.

* 4E	GET OPTION TABLE
*
*	Return a 12-byte option block.

* 4F	PUT OPTION TABLE
*
*	Accept a 12-byte option block and change to the indicated
*	configuration.

* 48	SET SIO SPEED
*
*	Change SIO baud rate to the rate specified by auxiliary byte
*	#1 (00 = 19200 baud, 01 = 38400 baud).


**	STATUS BLOCK FORMAT
*
*	Byte 1: Controller status
*		Bit 0 = 1 Invalid command frame
*		Bit 1 = 1 Invalid data frame
*		Bit 2 = 1 Operation Unsuccessful
*		Bit 3 = 1 Write protected (valid only after a write)
*		Bit 4 = 1 Motor on
*		Bits 5-7 = Device configuration
*			7 6 5
*			0 0 0  SS/SD
*			0 0 1  SS/DD
*			0 1 0  Reserved
*			0 1 1  Reserved
*			1 0 0  SS/2D
*			1 0 1  DS/DD
*			1 1 0  Reserved
*			1 1 1  Reserved
*
*	Byte 2 = 1770 status register (inverted).
*
*	Byte 3 = FORMAT timeout (= $78).
*
*	Byte 4 = Zero.


**	OPTION TABLE FORMAT
*
*	Byte 1 = Number of tracks (= 28).
*
*	Byte 2 = Step rate (= 00).
*		00 = 6 ms
*		01 = 12 ms
*		02 = 20 ms
*		03 = 30 ms
*
*	Byte 3 = Number of sectors per track (high).
*	Byte 4 = Number of sectors per track (low).
*
*	Byte 5 = Number of sides:
*		00 = single sided
*		01 = double sided
*
*	Byte 6 = Recording mode:
*		00 = FM
*		04 = MFM
*
*	Byte 7 = Number of bytes per sector (high).
*	Byte 8 = Number of bytes per sector (low).
*
*	Byte 9 = Drive present:
*		00 = not on line
*		01 = on line
*
*	Byte 10 = SIO bus speed:
*		41 = 19200 baud
*		48 = 38400 baud
*
*	Byte 11 = Reserved.
*	Byte 12 = Reserved.
*
*
*	NOTE: Bytes 1, 2, 3, 9, 10, 11, and 12 are ignored by the PUT
*	OPTION TABLE command.  The remaining bytes are used to set
*	the drive configuration, as follows:
*
*	IF bytes 5-6 = 00 and 00 AND the drive is a minifloppy,
*	THEN set SS/SD configuration;
*	ELSE IF bytes 4-8 = 1A, 00, 04, 00, and 80 respectively,
*	     AND the drive is a minifloppy,
*	THEN set SS/2D configuration;
*	ELSE IF bytes 5-6 = 00 and 04 respectively,
*	THEN set SS/DD configuration;
*	ELSE IF bytes 5-6 = 01 and 04 AND drive has two heads,
*	THEN set DS/DD configuration;
*	ELSE return error and leave configuration unchanged.


**	SIO TIMING

*	At 19200 baud, one bit time is 27.78 CPU cycles (which we
*	take as 28 CPU cycles).  At 38400 baud, one bit time is
*	13.89 CPU cycles (which we take as 14 CPU cycles).

*	The controller is responsible for generating the following
*	delays:
*	
*	t2 = delay between raising of COMMAND and transmission of ACK
*		t2 min = 0 us
*		t2 max = 16 ms
*
*	t4 = delay between transmission of the last bit of the data
*	     frame by the computer and transmission of first bit of
*	     ACK by the drive
*		t4 min = 850 us
*		t4 max = 16 ms
*
*	t5 = delay between transmission of last bit of ACK and
*	     transmission of first bit of CMPLT
*		t5 min = 250 us (19200 baud), 750 us (38400 baud)
*		t5 max = 255 sec (handler dependent)

*	The computer generates the following delays:
*
*	t0 = delay between lowering of COMMAND and transmission of
*	     first bit of command frame
*		t0 min = 750 us
*		t0 max = 1600 us
*
*	t1 = delay between transmission of last bit of command frame
*	     and raising of COMMAND
*		t1 min = 650 us
*		t1 max = 950 us
*
*	t3 = delay between receipt of last bit of ACK and transmission
*	     of first bit of data frame by the computer
*		t3 min = 1000 us
*		t4 max = 1800 us

	SUBTTL	'EQUATES'


**	TIMING EQUATES

CLOCK	EQU	375		;CPU CYCLE TIME IN UNITS OF 5 NS

US500	EQU	[[50000/CLOCK]-1]*2	;LOOP COUNT FOR 500 US DELAY
US200	EQU	[[20000/CLOCK]-1]*2	;LOOP COUNT FOR 200 US DELAY

TRIP	EQU	154		;TIMER OVERFLOW INTERVAL IN UNITS OF
				;100 US (= CLOCK*32*256*5/100000)

SEC3	EQU	30000/TRIP	;TRIP COUNT FOR 3 SECOND DELAY


**	WD1770 REGISTER SELECT EQUATES

WDCLR	EQU	$DC		;MASK TO CLEAR REGISTER SELECTION

WDRSTA	EQU	$20		;READ STATUS REGISTER
WDWCMD	EQU	$00		;WRITE COMMAND REGISTER
WDRTRK	EQU	$21		;READ TRACK REGISTER
WDWTRK	EQU	$01		;WRITE TRACK REGISTER
WDRSEC	EQU	$22		;READ SECTOR REGISTER
WDWSEC	EQU	$02		;WRITE SECTOR REGISTER
WDRDAT	EQU	$23		;READ DATA REGISTER
WDWDAT	EQU	$03		;WRITE DATA REGISTER


**	WD1770 COMMANDS

RDSEC	EQU	$80		;READ SECTOR
WRSEC	EQU	$A2		;WRITE SECTOR WITHOUT PRECOMP
WRSECP	EQU	$A0		;WRITE SECTOR WITH PRECOMP
RDADR	EQU	$C4		;READ ADDRESS WITH 30 MS DELAY
WRTRK	EQU	$F6		;WRITE TRACK WITHOUT PRECOMP
WRTRKP	EQU	$F4		;WRITE TRACK WITH PRECOMP
FRCINT	EQU	$D0		;FORCE INTERRUPT
REST	EQU	$00		;RESTORE WITH 6 MS STEP RATE
RESTF	EQU	$08		;RESTORE W/O MOTOR SPIN-UP
SEEK	EQU	$10		;SEEK WITH 6 MS STEP RATE


**	CONTROL LINES FOR PORT 1

INTRQ	EQU	$04		;1770 INTERRUPT REQUEST
DDEN	EQU	$08		;1770 DOUBLE DENSITY ENABLE
MR	EQU	$10		;1770 MASTER RESET
SSO	EQU	$40		;SIDE SELECT OUTPUT
SID	EQU	$80		;SERIAL INPUT DATA LINE
INP1	EQU	$04		;INPUT LINES ON PORT 1


**	CONTROL LINES FOR PORT 2

UNITMK	EQU	$C0		;MASK FOR UNIT NUMBER SWITCHES
UNIT1	EQU	$00		;UNIT 1 SELECT
UNIT2	EQU	$40		;UNIT 2 SELECT
UNIT3	EQU	$80		;UNIT 3 SELECT
UNIT4	EQU	$C0		;UNIT 4 SELECT
INP2	EQU	$80		;INPUT LINES FOR PORT 2


**	COMMUNICATION VALUES

ACK	EQU	$41		;ACKNOWLEDGE
COMPLT	EQU	$43		;SUCCESSFUL COMPLETION
ERROR	EQU	$45		;COMPLETION WITH ERROR
NAK	EQU	$4E		;NOT ACKNOWLEDGE
HSACK	EQU	$48		;FAST MODE ACKNOWLEDGE


**	ACTIVITY CODES

ACLR	EQU	$F8		;MASK TO CLEAR ACTIVITY CODE

AIDLE	EQU	$00		;IDLE
AGET	EQU	$01		;GET SECTOR
APUT	EQU	$02		;PUT SECTOR
APUTV	EQU	$03		;PUT SECTOR WITH VERIFY
APOPT	EQU	$04		;PUT OPTION TABLE
AMISC	EQU	$05		;MISCELLANEOUS DISK OPERATION
AOTHER	EQU	$07		;OTHER

	ASSERT	AIDLE=0		;LEGAL ASSUMPTION


**	HEAD POSITION KNOWN BIT

HKNOWN	EQU	$08		;HEAD POSITION KNOWN


**	CONFIGURATION CODES

CFCLR	EQU	$CF		;MASK TO CLEAR CONFIGURATION CODE
SECTSZ	EQU	$20		;MASK TO READ SECTOR SIZE BIT

CFSSSD	EQU	$20		;SS/SD
CFSS2D	EQU	$30		;SS/2D
CFSSDD	EQU	$00		;SS/DD
CFDSDD	EQU	$10		;DS/DD


**	INTERNAL RAM ADDRESSES

CSTAT	EQU	$FF		;CONTROLLER STATUS
HSTAT	EQU	$FE		;HARDWARE STATUS
CDEVIC	EQU	$FD		;COMMAND FRAME DEVICE NUMBER
CCOMND	EQU	CDEVIC-1	;COMMAND FRAME COMMAND
CAUX1	EQU	CCOMND-1	;COMMAND FRAME AUXILIARY #1
CAUX2	EQU	CAUX1-1		;COMMAND FRAME AUXILIARY #2

	SUBTTL	'MACROS'


**	JDRQ - JUMP IF DRQ

JDRQ	MACRO	ADDR
%L	JT1	%1
	ENDM


**	JNDRQ - JUMP IF NOT DRQ

JNDRQ	MACRO	ADDR
%L	JNT1	%1
	ENDM


**	SETREG - SET WD1770 REGISTER

SETREG	MACRO	REG
%L	ANLI	P1,WDCLR
	ORLI	P1,WD%1
	ENDM


**	JSOD - JUMP IF SERIAL OUTPUT DATA HIGH

JSOD	MACRO	ADDR
%L	JNT0	%1
	ENDM


**	JNSOD - JUMP IF SERIAL OUTPUT DATA LOW

JNSOD	MACRO	ADDR
%L	JT0	%1
	ENDM


**	CLRSID - CLEAR SERIAL INPUT DATA

CLRSID	MACRO
%L	ORLI	P1,SID
	ENDM


**	SETSID - SET SERIAL INPUT DATA

SETSID	MACRO
%L	ANLI	P1,$FF-SID
	ENDM


**	WINT - WAIT FOR INTERRUPT

WINT	MACRO
%L
MM%K	JUMP	MM%K
	ENDM


**	LNGJMP - LONG JUMP

LNGJMP	MACRO	ADDR
%L	DB	$E5 OR [[[%1]&$0800]/$80]
	JUMP	%1
	ENDM


**	PAUSE - DELAY FOR A FEW CPU CYCLES

PAUSE	MACRO	REGISTER,CYCLES
%L
	IF	[%2]&1
	NOPP
	ENDIF
	IF	[[%2]&$FFFE]=2
	NOPP
	NOPP
	ENDIF
	IF	[%2]>3
	MOVI	%1,[[%2]/2]-1
MM%K	DJNZ	%1,MM%K
	ENDIF
	ENDM


**	LNGCAL - LONG CALL

LNGCAL	MACRO	ADDR
%L	DB	$E5 OR [[[%1]&$0800]/$80]
	CALL	%1
	DB	$E5 OR [[[*]&$0800]/$80]
	ENDM

	SUBTTL	'COMMAND FRAME PROCESSOR'

	ORG	$1000

**	RESET COMES HERE

	LNGJMP	RESET		;GO DO RESET ROUTINE


**	EXTERNAL INTERRUPT COMES HERE

	ASSERT	*=$1003

	LNGJMP	COMPRO		;GO TO COMMAND FRAME PROCESSOR
	NOPP

**	TIMER INTERRUPT COMES HERE

	ASSERT	*=$1007

;	THIS CODE HANDLES TIMEOUTS WHICH OCCUR DURING RECEPTION OF
;	A DATA FRAME FROM THE SERIAL BUS.

	SEL	MB0
	DIS	TCNTI
	STOP	T
	CALL	ENABLE		;RESET INTERRUPT-IN-PROGRESS FLIP-FLOP

;	JUMP	INVDF		;INDICATE INVALID DATA FRAME


**	INVDF - PROCESS INVALID DATA FRAME

INVDF
	MOVI	R0,CSTAT
	MOVI	XR0,$02		;INVALID DATA FRAME

;	JUMP	IDLE		;GO TO IDLE STATE


**	IDLE - ENTER IDLE STATE

IDLE
	MOVI	R7,SEC3		;3 SECOND TRIP COUNT
	STRT	T
	ANLI	P2,ACLR		;INDICATE IDLE STATE
IDLE1
	JTF	IDLE2		;IF TIMER TRIPPED
	JUMP	IDLE1		;WAIT FOR TRIP
IDLE2
	DJNZ	R7,IDLE1	;IF 3 SECONDS NOT ELAPSED
IDLE4
	SETREG	RSTA
	MOVX	A,XR0		;READ THE STATUS REGISTER
	JB7	IDLE3		;IF MOTOR ON
	WINT			;MOTOR OFF, SO WAIT FOR INTERRUPT

;	MOTOR TIMED OUT, SO RESET 1770 TO FORCE MOTOR OFF

IDLE3
	ANLI	P2,$FF-HKNOWN	;INDICATE HEAD POSITION UNKNOWN
	ANLI	P1,$FF-MR	;SEND RESET SIGNAL
	PAUSE	R0,US500	;DELAY 500 US
	ORLI	P1,MR		;REMOVE RESET SIGNAL
	PAUSE	R0,US500
	SETREG	WCMD
	MOVI	A,FRCINT
	MOVX	XR0,A		;INITIATE FORCED INTERRUPT
	PAUSE	R0,US200	;DELAY 200 US

;	ENTRY POINT FOR RETURN FROM INTERRUPT

INTRET
	EN	I
	MOV	A,R7
	JNZ	IDLE1		;IF TIMING IN PROGRESS
	JUMP	IDLE4		;MAKE SURE MOTOR IS OFF


**	ENABLE - RESET THE INTERRUPT-IN-PROGRESS FLIP-FLOP

ENABLE
	RETR


**	COMPRO - COMMAND FRAME PROCESSOR

COMPRO

;	GET OFF THE SERIAL BUS

	SETSID

;	RESET THE INTERRUPT-IN-PROGRESS FLIP-FLOP

	DIS	TCNTI
	DIS	I
	CALL	ENABLE

;	CHECK TO SEE IF WE WERE IN MID-OPERATION

	IN	A,P2
	ANLI	A,$FF-ACLR
	JZ	COM1		;IF NOT IN MID-OPERATION

;	CONSTRUCT PSEUDO-STATUS IN CSTAT

	MOVI	R0,CSTAT
	MOVI	XR0,$04		;STATUS FOR UNSUCCESSFUL OPERATION

;	SET UP MOTOR TIMEOUT

	MOVI	R7,SEC3		;3 SECOND TRIP COUNT
	STRT	T		;START THE TIMER GOING

;	INDICATE IDLE STATE

	ANLI	P2,ACLR

;	INITIATE A FORCED INTERRUPT TO THE 1770 IF NECESSARY

	XRLI	A,APOPT
	JZ	COM1		;IF PUT OPTION TABLE
	XRLI	A,APOPT XOR AOTHER
	JZ	COM1		;IF OTHER NON-DISK OPERATION

	SETREG	WCMD
	MOVI	A,FRCINT
	MOVX	XR0,A

;	INITIALIZE FOR COMMAND FRAME RECEPTION

COM1
	MOVI	R0,CDEVIC	;STARTING ADDRESS
	MOVI	R1,0		;INIT CHECKSUM
	MOVI	R2,5		;NUMBER OF BYTES

;	CHECK COMMAND LINE

COM3
	JNI	COM5		;IF COMMAND LINE LOW
COM4
	JUMP	INTRET		;COMMAND FRAME SCREWED UP, RETURN

;	STORE BYTE OF COMMAND FRAME INTO RAM AND UPDATE CHECKSUM

COM2
	MOV	XR0,A
	DECR	R0
	ADD	A,R1
	ADDCI	A,0
	MOV	R1,A
	PAUSE	R3,12
	JNSOD	COM4		;IF STOP BIT IS BAD

;	WAIT FOR START BIT

COM5
	JSOD	COM3		;IF START BIT NOT DETECTED

;	INITIALIZE FOR DATA BYTE RECEPTION

	MOVI	A,$80
	PAUSE	R3,16		;DELAY 16 CPU CYCLES

;	LOOP TO READ IN BYTE BIT-BY-BIT

COM6
	PAUSE	R3,19		;DELAY 19 CPU CYCLES
	CLR	C
	JNSOD	COM7		;IF BIT IS A ZERO
	CPL	C
	JUMP	COM8
COM7
	PAUSE	R3,3		;EQUALIZE TIME OF THE TWO PATHS
COM8
	RRC	A		;SHIFT IN THE BIT
	JNC	COM6		;IF NOT DONE WITH 8 BITS

;	CHECK CHECKSUM

	DJNZ	R2,COM2		;IF CHECKSUM NOT RECEIVED YET
	XRL	A,R2
	JNZ	COM4		;IF CHECKSUM DOESN'T MATCH
	PAUSE	R3,15
	JNSOD	COM4		;IF STOP BIT IS WRONG
	JNI	COM9		;IF COMMAND LINE IS STILL LOW
	JUMP	COM4

;	CHECK TO SEE IF COMMAND IS FOR US

COM9
	MOVI	R1,$31		;DEVICE NUMBER FOR UNIT #1
	IN	A,P2		;READ DEVICE SELECT SWITCHES
	ANLI	A,UNITMK	;SELECT UNIT # BITS
	XRLI	A,UNIT1
	JZ	COM10		;IF UNIT 1
	INCR	R1
	XRLI	A,UNIT2 XOR UNIT1
	JZ	COM10		;IF UNIT 2
	INCR	R1
	XRLI	A,UNIT3 XOR UNIT2
	JZ	COM10		;IF UNIT 3
	INCR	R1
	XRLI	A,UNIT4 XOR UNIT3
	JNZ	COM4		;IF NOT UNIT 4
COM10
	MOVI	R0,CDEVIC
	MOV	A,XR0		;BRING IN DEVICE NUMBER
	XRL	A,R1
	JNZ	COM4		;IF COMMAND IS FOR A DIFFERENT UNIT

;	BRANCH TO VARIOUS COMMAND ROUTINES

	DECR	R0
	MOV	A,XR0		;BRING IN COMMAND
	XRLI	A,$52
	JNZ	COM20
	ORLI	P2,AGET		;SET ACTIVITY CODE FOR GET
	LNGJMP	GET		;GET SECTOR
COM20
	XRLI	A,$50 XOR $52
	JNZ	COM21
	ORLI	P2,APUT		;SET ACTIVITY CODE FOR PUT
	LNGJMP	PUT		;PUT SECTOR
COM21
	XRLI	A,$57 XOR $50
	JNZ	COM22
	ORLI	P2,APUTV	;SET ACTIVITY CODE FOR PUT/VERIFY
	LNGJMP	PUTV		;PUT/VERIFY SECTOR
COM22
	XRLI	A,$53 XOR $57
	JNZ	COM23
	ORLI	P2,AOTHER	;ACTIVITY CODE FOR 'OTHER'
	LNGJMP	STAT		;STATUS
COM23
	XRLI	A,$21 XOR $53
	JNZ	COM24
	ORLI	P2,AMISC	;ACTIVITY CODE FOR MISCELLANEOUS
	LNGJMP	FMT		;FORMAT
COM24
	XRLI	A,$4E XOR $21
	JNZ	COM25
	ORLI	P2,AOTHER	;ACTIVITY CODE FOR 'OTHER'
	LNGJMP	GOPT		;GET OPTION TABLE
COM25
	XRLI	A,$4F XOR $4E
	JNZ	COM26
	ORLI	P2,APOPT	;ACTIVITY CODE FOR PUT OPTION TABLE
	LNGJMP	POPT		;PUT OPTION TABLE
COM26
	XRLI	A,$48 XOR $4F
	JNZ	COM27
	ORLI	P2,AOTHER	;ACTIVITY CODE FOR 'OTHER'
	LNGJMP	SPEED		;SET SIO SPEED
COM27
	XRLI	A,$22 XOR $48
	JNZ	INVCF
	IN	A,P2
	ANLI	A,SECTSZ
	JZ	INVCF		;IF SECTOR SIZE IS 256
	ORLI	P2,AMISC	;ACTIVITY CODE FOR MISCELLANEOUS
	LNGJMP	FMTD		;FORMAT DUAL DENSITY


**	INVCF - PROCESS INVALID COMMAND FRAME

INVCF
	MOVI	R0,CSTAT
	MOVI	A,$01		;STATUS FOR INVALID COMMAND FRAME
	MOV	XR0,A		;SAVE IT

	MOVI	A,NAK
	LNGCAL	XBYTE		;SEND NAK
	JUMP	INTRET		;RETURN

	SUBTTL	'SUBROUTINES'


**	SENDAK - SEND ACKNOWLEDGE
*
*	EXIT CONDITIONS: A & R6 MODIFIED, INTERRUPTS ENABLED,
*		TIMER STOPPED AND CLEARED, TIMER OVERFLOW CLEARED,
*		RETRY FLAG CLEARED, 1770 MASTER RESET RAISED

SENDAK
	CLR	F1		;CLEAR RETRY FLAG
	STOP	T
	CLR	A
	MOV	T,A
	JTF	SNDAK1		;CLEAR TIMER OVERFLOW FLAG
SNDAK1
	MOVI	A,HSACK		;HIGH SPEED ACK
	JF0	SNDAK2		;IF HIGH SPEED ENABLED
	MOVI	A,ACK		;NORMAL ACK
SNDAK2
;	JUMP	XBYTE		;SEND BYTE, RETURN


**	XBYTE - SEND A BYTE OVER SERIAL BUS AT 19200 BUAD
*
*	ENTRY CONDITION: BYTE TO BE SENT IN A
*
*	EXIT CONDITION: R6 & A ARE MODIFIED, INTERRUPTS ARE ENABLED,
*		MASTER RESET TO 1770 IS RAISED

XBYTE
	JNI	XBYTE		;WAIT FOR COMMAND TO GO HIGH
	JNI	XBYTE		;JUST IN CASE
	ORLI	P1,MR		;RAISE 1770 RESET
	EN	I		;RE-ENABLE INTERRUPTS
	
	CLRSID			;SEND START BIT
	CLR	C
	CPL	C
	PAUSE	R6,3
XBYTE1
	PAUSE	R6,18		;DELAY 18 CPU CYCLES
	RRC	A		;SHIFT OUT NEXT BIT
	JC	XBYTE2		;IF BIT IS A ONE
	CLRSID			;SEND A ZERO
	JUMP	XBYTE3
XBYTE2
	SETSID			;SEND A ONE
	JUMP	XBYTE3		;EQUALIZE THE TIME AROUND BOTH PATHS
XBYTE3
	CLR	C
	JNZ	XBYTE1		;IF NOT DONE WITH 8 BITS

	RET

	SUBTTL	'SERIAL DATA FRAME RECEIVER'


**	RECV - RECEIVE DATA FRAME
*
*	ENTRY CONDITIONS: (NUMBER OF BYTES TO RECEIVE)-1 IN R0,
*		TIMER STOPPED
*
*	EXIT CONDITIONS: DATA FRAME IN INTERNAL RAM,
*		EXIT THRU INVDF IF CHECKSUM ERROR OR TIMEOUT.

RECV
	CLR	A		;INIT CHECKSUM
	CLR	C
	EN	TCNTI
	STRT	T		;BEGIN TIMEOUT FOR FIRST BYTE

;	READ IN THE FIRST [R0] BYTES

RECV5
	JNSOD	RECV19		;IF STOP BIT BAD
RECV1
	JSOD	RECV1		;IF START BIT NOT DETECTED
	ADD	A,R1
	ADDCI	A,0
	MOV	R1,A		;UPDATE CHECKSUM
	CLR	A
	MOV	T,A		;RESET THE TIMER
	MOVI	A,$80		;INIT FOR DATA BYTE RECEPTION

	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	JF0	RECV6		;DELAY 9 CPU CYCLES IN FAST MODE
	NOPP
	NOPP			;DELAY 11 CYCLES IN SLOW MODE

RECV2
	NOPP
	NOPP
	NOPP
	JF0	RECV6		;DELAY 5 CPU CYCLES IN FAST MODE
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP			;DELAY 19 CYCLES IN SLOW MODE

RECV6
	CLR	C
	JNSOD	RECV3		;IF BIT IS A ZERO
	CPL	C
	JUMP	RECV4
RECV3
	NOPP
	NOPP
	NOPP			;EQUALIZE TIME AROUND BOTH PATHS
RECV4
	RRC	A		;SHIFT IN THE BIT
	JNC	RECV2		;IF NOT DONE WITH 8 BITS
	XCH	A,XR0		;STORE BYTE IN RAM

	NOPP
	JF0	RECV7		;DELAY 3 CYCLES IN FAST MODE
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP			;DELAY 17 CYCLES IN SLOW MODE

RECV7
	DJNZ	R0,RECV5	;IF MORE BYTES TO DO
	JNSOD	RECV19		;IF STOP BIT BAD

;	BRING IN THE LAST DATA BYTE

RECV8
	JSOD	RECV8		;IF START BIT NOT DETECTED
	ADD	A,R1
	ADDCI	A,0
	MOV	R0,A		;UPDATE CHECKSUM
	CLR	A
	MOV	T,A		;RESET THE TIMER
	MOVI	A,$80		;INIT FOR DATA BYTE RECEPTION

	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	JF0	RECV9		;DELAY 9 CPU CYCLES IN FAST MODE
	NOPP
	NOPP			;DELAY 11 CYCLES IN SLOW MODE

RECV10
	NOPP
	NOPP
	NOPP
	JF0	RECV9		;DELAY 5 CPU CYCLES IN FAST MODE
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP			;DELAY 19 CYCLES IN SLOW MODE

RECV9
	CLR	C
	JNSOD	RECV11		;IF BIT IS A ZERO
	CPL	C
	JUMP	RECV12
RECV11
	NOPP
	NOPP
	NOPP			;EQUALIZE TIME AROUND BOTH PATHS
RECV12
	RRC	A		;SHIFT IN THE BIT
	JNC	RECV10		;IF NOT DONE WITH 8 BITS
	XCH	A,R0		;STORE BYTE IN RAM

	NOPP
	NOPP
	NOPP
	JF0	RECV13		;DELAY 5 CYCLES IN FAST MODE
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP			;DELAY 19 CYCLES IN SLOW MODE

;	COMPARE CALCULATED CHECKSUM TO RECEIVED CHECKSUM

RECV13
	JNSOD	RECV19		;IF STOP BIT IS BAD
RECV14
	JSOD	RECV14		;WAIT FOR START BIT
	STOP	T
	DIS	TCNTI
	ADD	A,R0
	ADDCI	A,0		;FINISH CHECKSUM CALCULATION
	CLR	C
	CPL	C

	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	JF0	RECV18		;DELAY 8 CYCLES IN FAST MODE
	NOPP			;DELAY 9 CYCLES IN SLOW MODE

RECV15
	NOPP
	NOPP
	NOPP
	NOPP
	JF0	RECV18		;DELAY 6 CPU CYCLES IN FAST MODE
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP
	NOPP			;DELAY 20 CYCLES IN SLOW MODE

RECV18
	RRC	A		;SHIFT OUT NEXT BIT TO COMPARE
	JC	RECV16		;IF BIT IS A ONE
	JNSOD	RECV17		;IF RECEIVED BIT IS ZERO
RECV19
	LNGJMP	INVDF		;GO PROCESS INVALID DATA FRAME
RECV16
	JNSOD	RECV19		;IF RECEIVED BIT IS ZERO, ERROR
RECV17
	CLR	C
	JNZ	RECV15		;IF DONE ALL BITS IN CHECKSUM

;	RETURN TO CALLER

	IN	A,P2
	ANLI	A,$FF-ACLR	;GET ACTIVITY CODE
	XRLI	A,APOPT
	JZ	RECV20		;IF PUT OPTION TABLE
	LNGJMP	PUTRR		;GO TO PUT RECEIVE RETURN
RECV20
	LNGJMP	POPTRR		;GO TO PUT OPTION TABLE RECIEVE RETURN

	SUBTTL	'STATUS'


**	STAT - GET STATUS

STAT
